/**
 * Classe che definisce le funzioni per il disegno di dati sulla mappa
 */
import { Draw, Select } from "ol/interaction";
import Button from "ol-ext/control/Button.js";
import MultiPolygon from "ol/geom/MultiPolygon.js";
import Feature from "ol/Feature.js";
import TextButton from "ol-ext/control/TextButton.js";
import DrawHole from "ol-ext/interaction/DrawHole";
import { singleClick } from "ol/events/condition";
import Tooltip from "ol-ext/overlay/Tooltip";
import Toggle from "ol-ext/control/Toggle";
import Bar from "ol-ext/control/Bar";
import "ol-ext/control/Bar.css";
import "ol-contextmenu/dist/ol-contextmenu.css";
import { drawLayer } from './map.js';
import WKT from "ol/format/WKT";
import {
	getComuneFromGeometry
} from '../utils/apicalls.js';

// Interazione selezione geometria
export const selectGeometryDrawed = new Select({
	condition: singleClick,
});

// Interazione disegno poligono
const drawPolygon = new Draw({
	source: drawLayer.getSource(),
	type: 'Polygon',
	stopClick: true
});

// Interazione disegno punto
const drawPoint = new Draw({
	source: drawLayer.getSource(),
	type: 'Point',
	stopClick: true
});

const drawEmptySection = new DrawHole({
	layers: [drawLayer]
});

// Creazione overlay tooltip area poligoni
const tooltip = new Tooltip();

// Bottone per selezionare le geometrie disegnate e salvarle
export const drawSelectBtn = new Toggle({
	html: '->',
	title: 'Seleziona geometrie',
	interaction: selectGeometryDrawed,
	active: false,
	disable: true
});

export function addToolbar(map, setDialogDrawDataOpen, setSelectedFeature, setAlertDialogOpen, setDescriptionAlertDialog) {

	// Attivata la modalita disegno disattivo i popup delle geometrie
	//selectGeometryPopup.setActive(false);

	map.addInteraction(drawPolygon);
	map.addInteraction(drawPoint);
	map.addInteraction(drawEmptySection);

	// Inizializzo la barra dei sottocontrolli del pulsante di selezione poligoni disegnati
	const subBarDrawSelectBtn = new Bar({
		disable: true,
		controls: [
			new Button({
				html: 'X',
				title: "Elimina",
				handleClick: function(e) {
					var features = selectGeometryDrawed.getFeatures();
					if (features.array_.length > 0) {
						//Elimina tutte le geometrie selezionate altrimenti nulla
						for (var i = 0, f; f = features.item(i); i++) {
							drawLayer.getSource().removeFeature(f);
						}
						selectGeometryDrawed.getFeatures().clear();
					} else {
						setAlertDialogOpen(true);
					}
				}
			}),
			new Button({
				html: 'V',
				title: "Salva",
				handleClick: async function() {
					var features = selectGeometryDrawed.getFeatures();

					if (features.array_.length > 0) {
						//disattivo l'interaction select
						selectGeometryDrawed.setActive(false);

						if (features.array_.length === 1) {

							//inizializzo la feature con il poligono disegnato
							const feature = features.array_[0];

							var f = new WKT().writeGeometry(feature.getGeometry());

							var comune = await fetchDataComune(f);
							if (comune === "Sconosciuto") {
								setDescriptionAlertDialog("Errore, la geometria disegnata si trova all'esterno dei comuni del progetto VeBS!")
								setAlertDialogOpen(true);
							} else {
								setSelectedFeature(feature);
								setDialogDrawDataOpen(true);
								selectGeometryDrawed.getFeatures().clear();
							}
						} else {
							//Trasformo un array di feature selezionate in un multipoligono partendo dalla funzione clone()
							var polygons = new Array();
							for (var i = 0, f; f = features.item(i); i++) {
								var polygon = f.getGeometry().clone();
								polygons.push(polygon);
							}

							// Trasformo in array di coordinate
							const coords = new Array();
							for (var j = 0, p; p = polygons[j]; j++) {
								var coord = p.getCoordinates();
								coords.push(coord);
							}

							// Creo la nuova MultiPolygon
							const multiPolygon = new MultiPolygon(coords);

							// Creo una nuova feature con il multipoligono
							const feature = new Feature(multiPolygon)

							var f = new WKT().writeGeometry(feature.getGeometry());

							var comune = fetchDataComune(feature);
							if (comune === "Sconosciuto") {
								setDescriptionAlertDialog("Errore, la geometria disegnata si trova all'esterno dei comuni del progetto VeBS!")
								setAlertDialogOpen(true);
							} else {
								//Salva tutte le geometrie disegnate
								setSelectedFeature(feature);
								setDialogDrawDataOpen(true);
								selectGeometryDrawed.getFeatures().clear();
							}
						}
					} else {
						setDescriptionAlertDialog("Errore, la geometria disegnata deve essere selezionata!!")
						setAlertDialogOpen(true);
					}
				}
			})
		]
	});

	// Creazione di una toolbar
	const toolbar = new Bar({
		toggleOne: true,
		source: drawLayer.getSource(),
	});

	// Bottone disegno poligoni
	const drawPolygonBtn = new Toggle({
		html: '+',
		title: 'Disegna poligono',
		interaction: drawPolygon,
		// Opzioni associate al disegno delle geometrie
		bar: new Bar({
			source: drawLayer.getSource(),
			controls: [
				// Opzione per l'eliminazione dell'ultimo punto disegnato
				new TextButton({
					html: 'indietro',
					title: "Elimina l'ultimo vertice disegnato",
					handleClick: function() {
						if (drawPolygon) drawPolygon.removeLastPoint();
					}
				})
			]
		})
	});

	// Bottone disegno geometrie puntiformi
	const drawPointBtn = new Toggle({
		html: 'T',
		title: 'Disegna punto',
		interaction: drawPoint,
		active: false
	});

	// Bottone per disegno sezione vuote nei poligoni disegnati
	const drawHolePolygonBtn = new Toggle({
		html: 'O',
		title: 'Disegna sezione vuota (dopo aver selezionato un poligono)',
		interaction: drawEmptySection,
		active: false,
		// Opzioni associate al disegno delle geometrie
		bar: new Bar({
			controls: [
				// Opzione per l'eliminazione dell'ultimo punto disegnato
				new TextButton({
					html: 'indietro',
					title: "Elimina l'ultimo vertice",
					handleClick: function() {
						if (drawEmptySection) drawEmptySection.removeLastPoint();
					}
				}),
			]
		})
	});

	drawSelectBtn.setSubBar(subBarDrawSelectBtn);

	// Setto il pulsante per il disegno punti disabilitato se è stato disegnato un poligono
	drawPolygon.on('drawend', () => {
		drawSelectBtn.setDisable(false);
	});

	// Setto il pulsante per il disegno poligoni disabilitato se è stato disegnato un punto
	drawPoint.on('drawend', () => {
		drawSelectBtn.setDisable(false);
	});
	
	addMeasureTooltips(map);

	toolbar.addControl(drawPolygonBtn);
	toolbar.addControl(drawPointBtn);
	toolbar.addControl(drawHolePolygonBtn);
	toolbar.addControl(drawSelectBtn);

	map.addControl(toolbar);

	return toolbar;

}


export function removeToolbar(map, toolbar) {

	map.removeControl(toolbar);
	map.removeInteraction(drawPolygon);
	map.removeInteraction(drawPoint);
	map.removeInteraction(drawEmptySection);
	map.removeOverlay(tooltip);
}


function addMeasureTooltips(map) {

	map.addOverlay(tooltip);
	// Gestione tooltip con interazione disegno poligoni
	drawPolygon.on('drawstart', tooltip.setFeature.bind(tooltip));
	drawPolygon.on(['change:active', 'drawend'], tooltip.removeFeature.bind(tooltip));

}

// funzione per la fetch del comune di riferimento alla geometria
async function fetchDataComune(geomWKT) {

	try {
		const comuneResp = await getComuneFromGeometry(geomWKT);
		const comuneJson = await comuneResp.json();

		if (comuneJson.length !== 0)
			return comuneJson[0].comune;
		else {
			return "Sconosciuto";
		}
	} catch (err) {
		console.error("Errore nel fetch dei dati del comune:", err);
	}
}
