import React, { useState, useEffect, useRef } from 'react';
import { Button } from '../../components/ui/button';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '../../components/ui/sheet';
import { ScrollArea } from '../../components/ui/scroll-area';
import { Separator } from '../../components/ui/separator';
import { Badge } from '../../components/ui/badge';
import { Input } from '../../components/ui/input';
import {
	Menu,
	Map,
	Settings,
	Home,
	Navigation,
	PencilRuler,
	ChevronLeft,
	ChevronRight,
	ChartPie,
	Database,
	Check,
	ChevronsUpDown,
	MapPin,
	Trash2,
	Lightbulb
} from 'lucide-react';
import { cn } from "../../lib/utils"
import {
	Command,
	CommandEmpty,
	CommandGroup,
	CommandInput,
	CommandItem,
	CommandList,
} from "../../components/ui/command"
import {
	Popover,
	PopoverContent,
	PopoverTrigger,
} from "../../components/ui/popover"

import { addToolbar, removeToolbar } from '../utils/draw_feature'
import { addPopup, removePopup } from '../utils/edit_feature'
import { addContextMenu, calculatePath, changeItemRemovePathFunction, removePath } from '../utils/directions_feature'
import { addOverlayStats, addSelectInteraction, filterLayers, removeOverlay } from '../utils/stats_feature'
import MapComponent from './MapComponent'
import {
	CZLayerAreeVerdiComune,
	layerAreeVerdiCasalecchioComune,
	layerAreeVerdiOpenStreetMap,
	layerIngressi,
	CZLayerCentroidi,
	layerComuneAquila,
	layerComuneCatanzaro,
	layerComuneCasalecchio,
	layerComuneTerracina,
	layerComunePescara,
	layerSezioniCensuarie
} from "../utils/map.js"

//mappa comuni
const municipalities = [
	{
		value: "tutti",
		label: "Tutti",
	},
	{
		value: "aquila",
		label: "L'Aquila",
	},
	{
		value: "catanzaro",
		label: "Catanzaro",
	},
	{
		value: "terracina",
		label: "Terracina",
	},
	{
		value: "casalecchio",
		label: "Casalecchio di Reno",
	},
	{
		value: "pescara",
		label: "Pescara",
	},
]

//array layers per comune
const layers = {
	tutti: [CZLayerAreeVerdiComune,
		CZLayerCentroidi,
		layerAreeVerdiCasalecchioComune,
		layerAreeVerdiOpenStreetMap,
		layerIngressi,
		layerComuneAquila,
		layerComuneCatanzaro,
		layerComuneCasalecchio,
		layerComuneTerracina,
		layerComunePescara,
		layerSezioniCensuarie],
	catanzaro: [CZLayerAreeVerdiComune,
		CZLayerCentroidi,
		layerComuneCatanzaro,]
}

// Componente Header
const Header = ({
	onMenuClick,
	isSidebarOpen,
	activeMenuItem,
	map,
	startingPoint,
	setStartingPoint,
	profile,
	setProfile,
	municipalityValue,
	setMunicipalityValue,
	municipalityOpen,
	setMunicipalityOpen,
	dynamicLayers,
	setDynamicLayers,
	typeGraphic,
	setTypeGraphic,
	typeTarget,
	setTypeTarget,
	setStatsOverlayOpen,
	setDataStats,
	selectedFeatureStats,
	alertDialogOpen,
	setAlertDialogOpen,
	descriptionAlertDialog,
	setDescriptionAlertDialog,
	typeStatistics,
	setTypeStatistics
}) => {
	const [searchQuery, setSearchQuery] = useState('');
	const [numVisibleLayers, setNumVisibleLayers] = useState(0);
	const [currentZoom, setCurrentZoom] = useState(0);
	const [startingPointCoords, setStartingPointCoords] = useState("");

	// Aggiorna layer in base al comune scelto
	useEffect(() => {
		if (!map) return;
		if (!municipalityValue) return;

		const allMapLayers = map.getLayers().getArray();
		const newLayers = layers[municipalityValue] || [];

		// Rimuovi i layer dinamici che NON sono più nel nuovo set
		dynamicLayers.forEach(layer => {
			if (!newLayers.includes(layer)) {
				map.removeLayer(layer);
			}
		});

		//Aggiungi i layer nuovi che non sono già presenti
		newLayers.forEach(layer => {
			const alreadyInMap = allMapLayers.includes(layer);
			if (!alreadyInMap) {
				map.addLayer(layer);
			}
		});

		//Aggiorna lo stato con i layer dinamici correnti
		setDynamicLayers(newLayers);

	}, [map, municipalityValue]);

	// Aggiorna zoom
	useEffect(() => {
		if (!map) return;

		const updateZoom = () => {
			const zoom = map.getView().getZoom();
			setCurrentZoom(zoom || 0);
		};

		// Aggiorna subito
		updateZoom();

		// Listener per cambio zoom
		const view = map.getView();
		const zoomKey = view.on('change:resolution', updateZoom);

		return () => {
			view.un('change:resolution', updateZoom);
		};
	}, [map]);

	// Aggiorna layer visibili
	useEffect(() => {
		if (!map) return;

		const updateLayers = () => {
			const allLayers = map.getLayers().getArray();
			const visibleCount = allLayers.filter(layer => {
				try {
					return layer.getVisible();
				} catch (e) {
					return false;
				}
			}).length;
			setNumVisibleLayers(Math.max(0, visibleCount - 1)); // -1 per escludere draw layer
		};

		// Aggiorna subito
		updateLayers();

		// Listener per cambio layer
		const layers = map.getLayers();

		// Listener per ogni layer esistente
		const updateVisibility = () => {
			updateLayers();
		};

		layers.forEach(layer => {
			layer.on('change:visible', updateVisibility);
		});

		// Listener per aggiunta/rimozione layer
		layers.on('add', updateVisibility);
		layers.on('remove', updateVisibility);

		return () => {
			layers.forEach(layer => {
				layer.un('change:visible', updateVisibility);
			});
			layers.un('add', updateVisibility);
			layers.un('remove', updateVisibility);
		};
	}, [map]);

	// Aggiorna coordinate punto di partenza
	useEffect(() => {
		if (!startingPoint) {
			setStartingPointCoords("");
			return;
		}

		try {
			const coords = startingPoint.getGeometry().getCoordinates();
			setStartingPointCoords(`${coords[0].toFixed(4)}, ${coords[1].toFixed(4)}`);
		} catch (error) {
			console.error("Errore calcolo coordinate:", error);
			setStartingPointCoords("Errore coordinate");
		}
	}, [startingPoint]);

	// Rendering del contenuto contestuale
	const renderContextualContent = () => {
		switch (activeMenuItem) {
			case 'home':
				return (
					<div className="flex items-center gap-8">
						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Layer Attivi</span>
							<div className="flex items-center gap-2">
								<Badge variant="outline" className="text-base font-semibold">
									{numVisibleLayers}
								</Badge>
							</div>
						</div>

						<Separator orientation="vertical" className="h-10" />

						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Livello Zoom</span>
							<span className="text-base font-semibold text-gray-900">
								{currentZoom.toFixed(1)}
							</span>
						</div>

						<Separator orientation="vertical" className="h-10" />

						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Comune Selezionato</span>
							<Popover open={municipalityOpen} onOpenChange={setMunicipalityOpen}>
								<PopoverTrigger asChild>
									<Button
										variant="ghost"
										role="combobox"
										size="sm"
										aria-expanded={municipalityOpen}
										className="h-8 justify-between hover:bg-gray-100 font-semibold"
									>
										{municipalityValue
											? municipalities.find((m) => m.value === municipalityValue)?.label
											: "Tutti"}
										<ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
									</Button>
								</PopoverTrigger>
								<PopoverContent className="w-[200px] p-0">
									<Command>
										<CommandInput placeholder="Cerca comune..." className="h-9" />
										<CommandList>
											<CommandEmpty>Nessun comune trovato.</CommandEmpty>
											<CommandGroup>
												{municipalities.map((municipality) => (
													<CommandItem
														key={municipality.value}
														value={municipality.value}
														onSelect={(currentValue) => {
															setMunicipalityValue(currentValue === municipalityValue ? "" : currentValue)
															setMunicipalityOpen(false)
														}}
													>
														{municipality.label}
														<Check
															className={cn(
																"ml-auto h-4 w-4",
																municipalityValue === municipality.value ? "opacity-100" : "opacity-0"
															)}
														/>
													</CommandItem>
												))}
											</CommandGroup>
										</CommandList>
									</Command>
								</PopoverContent>
							</Popover>
						</div>
					</div>
				);

			case 'directions':
				return (
					<div className="flex items-center gap-8">
						<div className="flex flex-col gap-1 min-w-[240px]">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Punto di Partenza</span>
							<div className="flex items-center gap-2">
								<MapPin className="h-4 w-4 text-blue-600 flex-shrink-0" />
								<span className="text-sm font-medium text-gray-900 truncate">
									{startingPointCoords || "Click tasto destro sull'edificio sulla mappa"}
								</span>
							</div>
						</div>

						<Separator orientation="vertical" className="h-10" />

						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Modalità di Trasporto</span>
							<select
								className="h-8 text-sm font-medium border-0 bg-transparent focus:ring-0 focus:outline-none cursor-pointer"
								value={profile}
								onChange={(e) => setProfile(e.target.value)}
							>
								<option value="driving-car">🚗 In Auto</option>
								<option value="wheelchair">♿ Sedia a Rotelle</option>
								<option value="foot-walking">🚶 A Piedi</option>
								<option value="cycling-road">🚴 In Bici</option>
							</select>
						</div>

						<Separator orientation="vertical" className="h-10" />

						<Button
							size="default"
							disabled={!startingPoint}
							onClick={() => {
								if (startingPoint) {
									var layerActive = null;
									map.getLayers().forEach(layer => {
										if (layer.values_.title === "Percorso")
										layerActive = layer;
									});
									if (layerActive) {
										removePath(map, setStartingPoint)
										calculatePath(startingPoint, map, profile);
									} else
										calculatePath(startingPoint, map, profile);
								}
							}}
							className="h-10 px-6"
						>
							<Navigation className="h-4 w-4 mr-2" />
							Calcola Percorso
						</Button>

						{startingPoint && (
							<Button
								variant="outline"
								size="default"
								onClick={() => setStartingPoint(null)}
								className="h-10"
							>
								<Trash2 className="h-4 w-4 mr-2" />
								Cancella
							</Button>
						)}
					</div>
				);

			case 'draw':
				return (
					<div className="flex items-center gap-6">
						<div className="flex items-center gap-3 text-gray-600">
							<Lightbulb className="h-5 w-5" />
							<span className="text-sm font-medium">
								Utilizza gli strumenti della toolbar per disegnare nuove geometrie sulla mappa
							</span>
						</div>
					</div>
				);

			case 'edit':
				return (
					<div className="flex items-center gap-6">
						<div className="flex items-center gap-3 text-gray-600">
							<Lightbulb className="h-5 w-5" />
							<span className="text-sm font-medium">
								Clicca su una geometria esistente per modificarne i dati
							</span>
						</div>
					</div>
				);

			case 'stats':
				return (
					<div className="flex items-center gap-8">
						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Target statistica</span>
							<select className="h-8 text-sm font-medium border-0 bg-transparent focus:ring-0 focus:outline-none cursor-pointer"
								value={typeTarget}
								onChange={(e) => (setTypeTarget(e.target.value), filterLayers(map, typeTarget))}>
								<option value="Comune">Comune</option>
								<option value="Sezione censuaria">Sezione Censuaria</option>
								<option value="Edificio popolazione">Edificio Popolazione</option>
								<option value="Area verde">Area Verde</option>
								<option value="Area blu">Area Blu</option>
							</select>
						</div>
						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Tipo di Grafico</span>
							<select className="h-8 text-sm font-medium border-0 bg-transparent focus:ring-0 focus:outline-none cursor-pointer"
								value={typeGraphic}
								onChange={(e) => setTypeGraphic(e.target.value)}>
								<option value="circle-packing">📊 Grafico circolare</option>
							</select>
						</div>
						<div className="flex flex-col gap-1">
							<span className="text-xs text-gray-500 uppercase tracking-wide">Tipo di Statistica</span>
							<select className="h-8 text-sm font-medium border-0 bg-transparent focus:ring-0 focus:outline-none cursor-pointer"
								value={typeStatistics}
								onChange={(e) => setTypeStatistics(e.target.value)}>
								<option value="numerousness">📊 Numerosità</option>
							</select>
						</div>

						<Separator orientation="vertical" className="h-10" />

						<Button size="default" className="h-10 px-6" onClick={() => { addOverlayStats(setStatsOverlayOpen, setDataStats, selectedFeatureStats, setAlertDialogOpen, setDescriptionAlertDialog) }}>
							<ChartPie className="h-4 w-4 mr-2" />
							Genera Statistiche
						</Button>
					</div>
				);

			case 'settings':
				return (
					<div className="flex items-center gap-6">
						<div className="flex items-center gap-3 text-gray-600">
							<Lightbulb className="h-5 w-5" />
							<span className="text-sm font-medium">
								Configura le impostazioni dell'applicazione e della mappa
							</span>
						</div>
					</div>
				);

			default:
				return null;
		}
	};

	return (
		<header className="border-b bg-white sticky top-0 z-50 shadow-sm">
			{/* Prima riga: Logo e ricerca */}
			<div className="h-16 flex w-full items-center px-6 border-b border-gray-100">
				<Button
					variant="ghost"
					size="sm"
					onClick={onMenuClick}
					className="mr-3 hover:bg-gray-100"
				>
					<Menu className="h-5 w-5" />
				</Button>

				<div className="flex items-center space-x-3">
					<div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg flex items-center justify-center">
						<Map className="h-5 w-5 text-white" />
					</div>
					<div className="flex flex-col">
						<h2 className="font-bold text-gray-900 text-base leading-tight">
							Progetto VeBS
						</h2>
					</div>
				</div>

				<div className="flex-1" />

				<div className="flex items-center space-x-3">
					<div className="relative">
						<Input
							placeholder="Cerca sulla mappa..."
							value={searchQuery}
							onChange={(e) => setSearchQuery(e.target.value)}
							className="w-80 pl-4 pr-4 h-10"
						/>
					</div>
				</div>
			</div>

			{/* Seconda riga: Contenuto contestuale */}
			<div className="min-h-[72px] flex w-full items-center px-6 py-4 bg-gradient-to-r from-gray-50 to-white">
				<div className="flex items-center gap-4 min-w-0 w-full">
					<div className="flex items-center gap-3 flex-shrink-0">
						{React.createElement(menuItems.find(item => item.id === activeMenuItem)?.icon || Home, {
							className: "h-5 w-5 text-blue-600"
						})}
						<div className="flex flex-col">
							<span className="text-sm font-semibold text-gray-900">
								{menuItems.find(item => item.id === activeMenuItem)?.label}
							</span>
							<span className="text-xs text-gray-500">
								{menuItems.find(item => item.id === activeMenuItem)?.description}
							</span>
						</div>
					</div>

					<Separator orientation="vertical" className="h-12 mx-2" />

					<div className="flex-1 min-w-0 overflow-x-auto">
						{renderContextualContent()}
					</div>
				</div>
			</div>
		</header>
	);
};

// Menu Items Configuration
const menuItems = [
	{
		id: 'home',
		label: 'Dashboard Mappa',
		icon: Home,
		description: 'Vista principale',
		badge: null
	},
	{
		id: 'directions',
		label: 'Visualizza accessibilità',
		icon: Navigation,
		description: 'Strumento di visualizzazione percorso',
		badge: null
	},
	{
		id: 'stats',
		label: 'Visualizza statistiche',
		icon: ChartPie,
		description: 'Strumento di visualizzazione statistiche',
		badge: null
	},
	{
		id: 'draw',
		label: 'Aggiungi Dati',
		icon: PencilRuler,
		description: 'Disegna e aggiungi geometrie',
		badge: 'ADMIN'
	},
	{
		id: 'edit',
		label: 'Modifica Dati',
		icon: Database,
		description: 'Aggiorna dati geometrie',
		badge: 'ADMIN'
	},
	{
		id: 'settings',
		label: 'Impostazioni',
		icon: Settings,
		description: 'Configurazione app',
		badge: 'ADMIN'
	}
];
// Componente Sidebar
const Sidebar = ({ isOpen, onClose, activeMenuItem, onMenuItemClick, setDialogDrawDataOpen, setAlertDialogOpen, setDescriptionAlertDialog, map, setSelectedFeature, selectedFeature, setStartingPoint, setStatsOverlayOpen, setDataStats, setSelectedFeatureStats, selectedFeatureStats, typeTarget }) => {
	const [isMobile, setIsMobile] = useState(false);

	const drawToolbar = useRef(null);
	const geometryPopup = useRef(null);
	const contextMenu = useRef(null);
	const overlayStats = useRef(null);

	useEffect(() => {
		const checkMobile = () => {
			setIsMobile(window.innerWidth < 768);
		};

		checkMobile();
		window.addEventListener('resize', checkMobile);
	}, []);

	// Funzione per gestire il click sugli item con logica specifica
	const handleMenuClick = (itemId) => {
		onMenuItemClick(itemId);

		//Logica specifica per ogni item
		if (itemId === "draw" && map) {
			if (geometryPopup.current !== null) {
				removePopup(map, geometryPopup.current);
				geometryPopup.current = null;
			} else if (overlayStats.current !== null) {
				removeOverlay(map, setDataStats, setStatsOverlayOpen, setAlertDialogOpen, setDescriptionAlertDialog, setSelectedFeatureStats);
				overlayStats.current = null;
			} else if (contextMenu.current !== null) {
				changeItemRemovePathFunction(map, setStartingPoint);
				contextMenu.current = null;
			}
			//inizializzo la toolbar per il disegno
			if (drawToolbar.current === null)
				drawToolbar.current = addToolbar(map, setDialogDrawDataOpen, setSelectedFeature, setAlertDialogOpen, setDescriptionAlertDialog);


		} else if (itemId === "edit" && map) {
			if (drawToolbar.current !== null) {
				removeToolbar(map, drawToolbar.current);
				drawToolbar.current = null;
			} else if (overlayStats.current !== null) {
				removeOverlay(map, setDataStats, setStatsOverlayOpen, setAlertDialogOpen, setDescriptionAlertDialog, setSelectedFeatureStats);
				overlayStats.current = null;
			} else if (contextMenu.current !== null) {
				changeItemRemovePathFunction(map, setStartingPoint);
				contextMenu.current = null;
			}
			//inizializzo il popup per la modifica dati
			if (geometryPopup.current === null)
				geometryPopup.current = addPopup(map);

		} else if (itemId === "directions" && map) {
			if (drawToolbar.current !== null) {
				removeToolbar(map, drawToolbar.current);
				drawToolbar.current = null;
			} else if (geometryPopup.current !== null) {
				removePopup(map, geometryPopup.current);
				geometryPopup.current = null;
			} else if (overlayStats.current !== null) {
				removeOverlay(map, setDataStats, setStatsOverlayOpen, setAlertDialogOpen, setDescriptionAlertDialog, setSelectedFeatureStats);
				overlayStats.current = null;
			}
			if (contextMenu.current === null) {
				contextMenu.current = addContextMenu(map, setStartingPoint);
			}

		} else if (itemId === 'stats') {
			if (drawToolbar.current !== null) {
				removeToolbar(map, drawToolbar.current);
				drawToolbar.current = null;
			} else if (geometryPopup.current !== null) {
				removePopup(map, geometryPopup.current);
				geometryPopup.current = null;
			} else if (contextMenu.current !== null) {
				changeItemRemovePathFunction(map, setStartingPoint);
				contextMenu.current = null;
			}
			if (overlayStats.current === null) {
				filterLayers(map, typeTarget);
				overlayStats.current = addSelectInteraction(map, setSelectedFeatureStats);
			}

		} else if (itemId === 'settings') {
			if (drawToolbar.current !== null) {
				removeToolbar(map, drawToolbar.current);
				drawToolbar.current = null;
			} else if (geometryPopup.current !== null) {
				removePopup(map, geometryPopup.current);
				geometryPopup.current = null;
			} else if (overlayStats.current !== null) {
				removeOverlay(map, setDataStats, setStatsOverlayOpen, setAlertDialogOpen, setDescriptionAlertDialog, setSelectedFeatureStats);
				overlayStats.current = null;
			} else if (contextMenu.current !== null) {
				changeItemRemovePathFunction(map, setStartingPoint);
				contextMenu.current = null;
			}

		} else {
			if (drawToolbar.current !== null) {
				removeToolbar(map, drawToolbar.current);
				drawToolbar.current = null;
			} else if (geometryPopup.current !== null) {
				removePopup(map, geometryPopup.current);
				geometryPopup.current = null;
			} else if (overlayStats.current !== null) {
				removeOverlay(map, setDataStats, setStatsOverlayOpen, setAlertDialogOpen, setDescriptionAlertDialog, setSelectedFeatureStats);
				overlayStats.current = null;
			}


		}

		// Chiudi sidebar su mobile dopo selezione
		if (isMobile) {
			onClose();
		}
	};

	return (
		<>
			{/* Desktop Sidebar */}
			<div
				className={`
						hidden md:flex md:flex-col md:fixed md:inset-y-0 md:top-[112px] md:z-40
						transform transition-transform duration-200 ease-in-out bg-white border-r shadow-sm
						${isOpen ? 'translate-x-0' : '-translate-x-full'}
					`}
				style={{ width: '280px', minWidth: '280px', maxWidth: '280px' }}
			>
				<div className="flex flex-col h-full p-4">
					<h3 className="font-semibold text-sm text-gray-700 mb-3">Menu Principale</h3>
					<ScrollArea className="flex-1">
						<div className="space-y-2">
							{menuItems.map((item) => (
								<MenuItem
									key={item.id}
									item={item}
									isActive={activeMenuItem === item.id}
									onClick={handleMenuClick}
								/>
							))}
						</div>
					</ScrollArea>
				</div>
			</div>

			{/* Mobile Sidebar */}
			<div className="md:hidden">
				<Sheet open={isOpen} onOpenChange={onClose}>
					<SheetContent side="left" className="p-0 w-[280px]">
						<SheetHeader className="p-4 border-b">
							<SheetTitle>Menu</SheetTitle>
							<SheetDescription>
								Seleziona una funzionalità
							</SheetDescription>
						</SheetHeader>
						<ScrollArea className="h-[calc(100vh-80px)] p-4">
							<div className="space-y-2">
								{menuItems.map((item) => (
									<MenuItem
										key={item.id}
										item={item}
										isActive={activeMenuItem === item.id}
										onClick={handleMenuClick}
									/>
								))}
							</div>
						</ScrollArea>
					</SheetContent>
				</Sheet>
			</div>
		</>
	);
};

// Componente per ogni voce di menu - TESTO A CAPO DENTRO LARGHEZZA FISSA
const MenuItem = ({ item, isActive, onClick }) => {
	const Icon = item.icon;

	return (
		<div
			className={`
				flex items-start space-x-3 px-3 py-2 rounded-lg cursor-pointer transition-colors
				${isActive
					? 'bg-blue-100 text-blue-700 border border-blue-200'
					: 'hover:bg-gray-100 text-gray-700'
				}
			`}
			style={{ width: '100%', maxWidth: '288px' }}
			onClick={() => onClick(item.id)}
		>
			<Icon className="h-5 w-5 flex-shrink-0 mt-0.5" />
			<div className="flex-1" style={{ maxWidth: 'calc(100% - 32px)' }}>
				<p className="text-sm font-medium whitespace-normal break-words leading-tight" style={{ wordWrap: 'break-word' }}>
					{item.label}
				</p>
				<p className="text-xs text-gray-500 whitespace-normal break-words leading-tight mt-0.5" style={{ wordWrap: 'break-word' }}>
					{item.description}
				</p>
			</div>
			{item.badge && (
				<Badge
					variant={item.badge === 'NEW' ? 'default' : 'secondary'}
					className="text-xs flex-shrink-0"
				>
					{item.badge}
				</Badge>
			)}
		</div>
	);
};

// Componente principale
export default function SidebarComponent() {

	// variabili di stato del modale di inserimento dati del disegno
	const [dialogDataOpen, setDialogDataOpen] = useState(false);
	const [dialogDrawDataOpen, setDialogDrawDataOpen] = useState(false);
	// variabili di stato dell'alert
	const [alertDialogOpen, setAlertDialogOpen] = useState(false);
	const [descriptionAlertDialog, setDescriptionAlertDialog] = useState("");
	// variabili di stato della funzione disegno
	const [selectedFeature, setSelectedFeature] = useState(null);
	// variabili di stato del modale selector
	const [selectorOpen, setSelectorOpen] = useState(false);
	const [data, setData] = useState(null);
	//variabili di stato per l'inizializzazione della mappa
	const [map, setMap] = useState(null);
	//variabili di stato per l'inizializzazione della funzionalità di calcolo percorsi
	const [startingPoint, setStartingPoint] = useState(null);
	//variabili di stato per il profilo di trasporto scelto dall'utente
	const [profile, setProfile] = useState('driving-car');
	//variabili di stato per la selezione del comune da visualizzare
	const [municipalityValue, setMunicipalityValue] = useState("tutti");
	const [municipalityOpen, setMunicipalityOpen] = useState(false);
	const [dynamicLayers, setDynamicLayers] = useState(layers.tutti);
	const [sidebarOpen, setSidebarOpen] = useState(false);
	const [activeMenuItem, setActiveMenuItem] = useState('home');
	//variabili di stato per la funzionalità di statistica
	const [typeTarget, setTypeTarget] = useState("comune");
	const [typeGraphic, setTypeGraphic] = useState("Grafico circolare");
	const [statsOverlayOpen, setStatsOverlayOpen] = useState(false);
	const [dataStats, setDataStats] = useState(null);
	const [selectedFeatureStats, setSelectedFeatureStats] = useState(null);
	const [typeStatistics, setTypeStatistics] = useState("Numerosità");

	//useRef per l'utilizzo delle funzionalità
	const drawToolbar = useRef(null);
	const geometryPopup = useRef(null);
	const contextMenu = useRef(null);
	const statsOverlay = useRef(null);

	const toggleSidebar = () => {
		setSidebarOpen(!sidebarOpen);
	};

	const handleMenuItemClick = (itemId) => {
		setActiveMenuItem(itemId);
		console.log(`🎯 Menu selezionato: ${itemId}`);
	};

	return (
		<div className="h-screen flex flex-col bg-gray-50">
			{/* Header */}
			<Header
				onMenuClick={toggleSidebar}
				isSidebarOpen={sidebarOpen}
				activeMenuItem={activeMenuItem}
				map={map}
				startingPoint={startingPoint}
				setStartingPoint={setStartingPoint}
				profile={profile}
				setProfile={setProfile}
				municipalityValue={municipalityValue}
				setMunicipalityValue={setMunicipalityValue}
				municipalityOpen={municipalityOpen}
				setMunicipalityOpen={setMunicipalityOpen}
				dynamicLayers={dynamicLayers}
				setDynamicLayers={setDynamicLayers}
				typeGraphic={typeGraphic}
				setTypeGraphic={setTypeGraphic}
				typeTarget={typeTarget}
				setTypeTarget={setTypeTarget}
				setStatsOverlayOpen={setStatsOverlayOpen}
				setDataStats={setDataStats}
				setSelectedFeature={setSelectedFeature}
				selectedFeature={selectedFeature}
				setSelectedFeatureStats={setSelectedFeatureStats}
				selectedFeatureStats={selectedFeatureStats}
				alertDialogOpen={alertDialogOpen}
				setAlertDialogOpen={setAlertDialogOpen}
				descriptionAlertDialog={descriptionAlertDialog}
				setDescriptionAlertDialog={setDescriptionAlertDialog}
				typeStatistics={typeStatistics}
				setTypeStatistics={setTypeStatistics}
			/>

			<div className="flex flex-1 relative">
				{/* Sidebar */}

				<Sidebar
					isOpen={sidebarOpen}
					onClose={() => setSidebarOpen(false)}
					activeMenuItem={activeMenuItem}
					onMenuItemClick={handleMenuItemClick}
					map={map}
					drawToolbar={drawToolbar}
					geometryPopup={geometryPopup}
					contextMenu={contextMenu}
					setStartingPoint={setStartingPoint}
					statsOverlay={statsOverlay}
					setStatsOverlayOpen={setStatsOverlayOpen}
					setDataStats={setDataStats}
					selectedFeature={selectedFeature}
					setSelectedFeature={setSelectedFeature}
					selectedFeatureStats={selectedFeatureStats}
					setSelectedFeatureStats={setSelectedFeatureStats}
					typeTarget={typeTarget}
					setDialogDrawDataOpen={setDialogDrawDataOpen}
				/>

				{/* Overlay per mobile */}
				{sidebarOpen && (
					<div
						className="fixed inset-0 bg-black bg-opacity-50 z-30 md:hidden"
						onClick={() => setSidebarOpen(false)}
					/>
				)}

				{/* Map Container */}
				<main className={`
          flex-1 relative transition-all duration-200 ease-in-out
          ${sidebarOpen ? 'md:ml-80' : 'md:ml-0'}
        `}>

					{/* Pulsante toggle sidebar floating (desktop) */}
					<Button
						variant="outline"
						size="sm"
						className={`
              absolute top-4 z-10 transition-all duration-200
              ${sidebarOpen ? 'left-[324px]' : 'left-4'}
              hidden md:flex
            `}
						onClick={toggleSidebar}
					>
						{sidebarOpen ? <ChevronLeft className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
					</Button>

					{/* Mappa OpenLayers */}
					<MapComponent className="w-full h-full"
						setMap={setMap}
						map={map}
						dialogDataOpen={dialogDataOpen}
						setDialogDataOpen={setDialogDataOpen}
						dialogDrawDataOpen={dialogDrawDataOpen}
						setDialogDrawDataOpen={setDialogDrawDataOpen}
						alertDialogOpen={alertDialogOpen}
						setAlertDialogOpen={setAlertDialogOpen}
						descriptionAlertDialog={descriptionAlertDialog}
						setDescriptionAlertDialog={setDescriptionAlertDialog}
						selectedFeature={selectedFeature}
						setSelectedFeature={setSelectedFeature}
						selectorOpen={selectorOpen}
						setSelectorOpen={setSelectorOpen}
						data={data}
						statsOverlayOpen={statsOverlayOpen}
						setStatsOverlayOpen={setStatsOverlayOpen}
						dataStats={dataStats}
						typeGraphic={typeGraphic}
					/>
					{/* Badge stato attivo */}
					<div className="absolute bottom-4 left-8 z-10">
						<Badge variant="secondary">
							Funzionalità Attiva: {menuItems.find(item => item.id === activeMenuItem)?.label}
						</Badge>
					</div>
				</main>
			</div>
		</div >
	);
}