/**
 * Classe che contiene la finestra di dialogo per l'inserimento a db dei poligoni disegnati
 */

import React, { useState, useEffect } from "react";
import WKT from "ol/format/WKT";

import { getArea } from 'ol/sphere.js';

import {
	FormDisegnoPoligono,
	FormDisegnoPunto
} from '../utils/modals.jsx';

import {
	getComuneFromGeometry,
	getAllAreeBluFromComune,
	getAllAreeVerdiFromComune
} from '../utils/apicalls.js';

import { useDialogStack } from './DialogStackProvider.jsx'

function DialogDrawDataComponent({ open, onClose, onSubmit, feature }) {

	let tipoGeometria = "";
	var tipo = null;
	var f;

	if (feature !== null)
		if (feature.feature !== undefined)
			f = feature.feature;
		else
			f = feature;
	else
		f = feature;

	if (f) {
		tipo = f.values_.geometry.getType();
		if (tipo === "Polygon" || tipo === "MultiPolygon")
			tipoGeometria = "Poligono";
		else
			tipoGeometria = "Punto";
	}

	// Inizializzo la varibile per la scelta tipologia geometria
	let tipiInserimentoGeometria = [
		{ id: "areablu", nome: "Area blu" },
		{ id: "polareaverde", nome: "Poligono area verde" }
	];

	const [formDrawGeometryData, setFormDrawGeometryData] = useState({});

	const [date, setDate] = useState(
		new Date()
	);

	const [openDatePicker, setOpenDatePicker] = useState(true);

	const [valueTipoGeometria, setValueTipoGeometria] = useState("");

	const [valueAreaBlu, setValueAreaBlu] = useState("");
	const [valueAreaVerde, setValueAreaVerde] = useState("");
	const [areeBlu, setAreeBlu] = useState([]);
	const [areeVerdi, setAreeVerdi] = useState([]);
	const [comune, setComune] = useState([]);
	const [dataLoaded, setDataLoaded] = useState(false);

	//variabili di stato del selector
	const [openSelector, setOpenSelector] = useState(false);

	//variabili per lo dialog stack
	const { push } = useDialogStack();

	useEffect(() => {
		const fetchData = async () => {
			let featureData = {
				nome: "",
				densita_alberi: 0,
				tipo_geometria: tipo === "Point" ? "Ingresso" : "",
				geometria: f ? new WKT().writeGeometry(f.getGeometry()) : "",
				area: f ? getArea(f.getGeometry()) : 0,
				comune: "",
				data: new Date(),
				area_blu: 0,
				area_verde: 0,
			};

			await fetchDataComune(setComune, featureData.geometria);
			(tipoGeometria === "Punto") ? await fetchDataAree(setAreeBlu, setAreeVerdi, comune) : null;

			setFormDrawGeometryData(prev => ({ ...prev, ...featureData }));
			setDataLoaded(true);
		};

		if (f) fetchData();
	}, [feature]);

	const handleChange = e => {
		const { name, value } = e.target;
		setFormDrawGeometryData(prev => ({ ...prev, [name]: value }));
	};

	const handleSubmit = async () => {

		const payload = {
			...formDrawGeometryData,
			data: formDrawGeometryData.data.toISOString().split("T")[0],
		};

		onSubmit(payload);

		//svuoto le variabili di stato
		setValueAreaBlu(null);
		setValueAreaVerde(null);
		setValueTipoGeometria(null);
		setFormDrawGeometryData({});
		setDataLoaded(false);
		setAreeBlu([]);
		setAreeVerdi([]);
		setComune(null);

		//chiudo la finestra di dialogo
		onClose();
	};

	const handleCancel = () => {
		//svuoto le variabili di stato
		setValueAreaBlu(null);
		setValueAreaVerde(null);
		setValueTipoGeometria(null);
		setFormDrawGeometryData({});
		setDataLoaded(false);
		setAreeBlu([]);
		setAreeVerdi([]);
		setComune(null);

		//chiudo la finestra di dialogo
		onClose();
	}

	if (!dataLoaded) return null;

	if (tipoGeometria === "Poligono")
		return (FormDisegnoPoligono(open, onClose, handleSubmit, handleCancel, formDrawGeometryData, handleChange, openDatePicker, setOpenDatePicker, date, setDate, tipiInserimentoGeometria,
			valueTipoGeometria, setValueTipoGeometria, comune));
	else
		return (FormDisegnoPunto(open, onClose, handleSubmit, handleCancel, formDrawGeometryData, handleChange, openDatePicker, setOpenDatePicker, date, setDate, areeBlu,
			areeVerdi, valueAreaBlu, setValueAreaBlu, valueAreaVerde, setValueAreaVerde, comune, push, setOpenSelector));

}

export default DialogDrawDataComponent;

// funzione per la fetch delle geometrie delle aree blu e verdi
async function fetchDataAree(setAreeBlu, setAreeVerdi, comune) {
	try {
		const areeBluResp = await getAllAreeBluFromComune(comune);
		const areeBluJson = await areeBluResp.json();
		const areeVerdiResp = await getAllAreeVerdiFromComune(comune);
		const areeVerdiJson = await areeVerdiResp.json();

		setAreeBlu(areeBluJson || []);
		setAreeVerdi(areeVerdiJson || []);
	} catch (err) {
		console.error("Errore nel fetch dei dati delle aree:", err);
	}
}

// funzione per la fetch del comune di riferimento alla geometria
async function fetchDataComune(setComune, geomWKT) {

	try {
		const comuneResp = await getComuneFromGeometry(geomWKT);
		const comuneJson = await comuneResp.json();

		let comuneValue = "Sconosciuto";
		if (comuneJson.length !== 0) {
			comuneValue = comuneJson[0].comune;
		}

		setComune(comuneValue);
		return comuneValue;
	} catch (err) {
		console.error("Errore nel fetch dei dati del comune:", err);
	}
}

