/**
 * Classe che contiene la finestra di dialogo per la modifica dei dati dell'elemento selezionato dall'utente
 */

import React, { useState, useEffect } from "react";
import WKT from "ol/format/WKT";

import {
	getAllAreeVerdiFromComune,
	getAllAreeBluFromComune,
	getComuneFromGeometry,
	getAreaBluFromId,
	getAreaVerdeFromId
} from '../utils/apicalls.js';

import {
	FormAreaBlu,
	FormAreaVerde,
	FormIngresso,
	FormPoligonoAreaVerde
} from '../utils/modals.jsx';

import { useDialogStack } from './DialogStackProvider.jsx'

function DialogDataComponent({ open, onClose, onSubmit, feature }) {

	let typeGeometry = "";

	if (feature)
		typeGeometry = getTypeGeometry(feature.id_);

	const [formGeometryData, setFormGeometryData] = useState({});
	const [date, setDate] = useState();

	const [openDatePicker, setOpenDatePicker] = useState(true);

	const [valueAreaBlu, setValueAreaBlu] = useState("");
	const [valueAreaVerde, setValueAreaVerde] = useState("");

	const [areeBlu, setAreeBlu] = useState([]);
	const [areeVerdi, setAreeVerdi] = useState([]);
	const [comune, setComune] = useState(null);

	var [area, setArea] = useState();
	const [areaButton, setAreaButton] = useState(true);

	const [poligonoChecked, setPoligonoChecked] = useState(false);

	//variabili di stato del selector
	const [openSelector, setOpenSelector] = useState(false);
	const [dataLoaded, setDataLoaded] = useState(false);

	//variabili per lo dialog stack
	const { push } = useDialogStack();


	useEffect(() => {

		var featureData;

		const fetchData = async () => {
			featureData = {
				id: (feature.values_.id === null) ? "" : feature.values_.id,
				nome: (feature.values_.nome === null) ? "" : feature.values_.nome,
				densita_alberi: (feature.values_.densita_alberi === null) ? 0 : feature.values_.densita_alberi,
				ubicazione: (feature.values_.ubicazione === null) ? 0 : feature.values_.ubicazione,
				geometria: feature.values_.geometry ? new WKT().writeGeometry(feature.values_.geometry) : "",
				area: feature.values_.area ? (setArea(feature.values_.area), setAreaButton(false)) : (setArea(null), setAreaButton(true)),
				comune: (feature.values_.comune === null) ? "" : feature.values_.comune,
				data: (feature.values_.data === null) ? new Date() : feature.values_.data,
				area_blu: (feature.values_.area_blu === null) ? 0 : (feature.values_.area_blu, await fetchDataAreaBlu(feature.values_.area_blu, setValueAreaBlu)),
				area_verde: (feature.values_.area_verde === null) ? 0 : (feature.values_.area_verde, await fetchDataAreaVerde(feature.values_.area_verde, setValueAreaVerde)),
				tipo: typeGeometry
			};

			await fetchDataComune(setComune, featureData.geometria);
			(typeGeometry === "ingresso") ? await fetchDataAree(setAreeBlu, setAreeVerdi, comune) : null;

			setFormGeometryData(prev => ({ ...prev, ...featureData }));
		};

		if (feature && !(feature.type === 'drawend')) fetchData();
		setDataLoaded(true);

	}, [feature]);

	const handleChange = e => {
		const { name, value } = e.target;
		setFormGeometryData(prev => ({ ...prev, [name]: value }));
	};

	const handleSubmit = async () => {

		const payload = {
			...formGeometryData,
			data: formGeometryData.data.toISOString().split("T")[0],
		};

		onSubmit(payload);
		
		//svuoto le variabili di stato
		setFormGeometryData({});
		setValueAreaBlu(null);
		setValueAreaVerde(null);
		typeGeometry="";
		setComune(null);
		setAreeBlu([]);
		setAreeVerdi([]);
		setDataLoaded(false);

		//chiudo la finestra di dialogo
		onClose();
	};

	const handleCancel = () => {
		//svuoto le variabili di stato
		setValueAreaBlu(null);
		setValueAreaVerde(null);
		setFormGeometryData({});
		setDataLoaded(false);
		typeGeometry="";
		setAreeBlu([]);
		setAreeVerdi([]);
		setComune(null);

		//chiudo la finestra di dialogo
		onClose();
	}

	if (!dataLoaded) return null;

	// Assegno il tipo di geometria
	if (typeGeometry === "areaverde")
		return (FormAreaVerde(open, onClose, handleSubmit, formGeometryData, handleChange, handleCancel, openDatePicker, setOpenDatePicker, date, setDate, area, setArea, areaButton, setAreaButton, setFormGeometryData, poligonoChecked, setPoligonoChecked));
	else if (typeGeometry === "areablu")
		return (FormAreaBlu(open, onClose, handleSubmit, formGeometryData, handleChange, handleCancel, openDatePicker, setOpenDatePicker, date, setDate, setFormGeometryData));
	else if (typeGeometry === "polareaverde")
		return (FormPoligonoAreaVerde(open, onClose, handleSubmit, formGeometryData,handleChange, handleCancel, openDatePicker, setOpenDatePicker, date, setDate, setFormGeometryData));
	else if (typeGeometry === "ingresso")
		return (FormIngresso(open, onClose, handleSubmit, formGeometryData, handleChange, handleCancel, openDatePicker, setOpenDatePicker, date, setDate, areeBlu,
			areeVerdi, valueAreaBlu, setValueAreaBlu, valueAreaVerde, setValueAreaVerde, push, setOpenSelector));
	else return (null);
}

export default DialogDataComponent;

// Funzione per il riconoscimento del tipo di geometria rispetto alla risposta di geoserver
function getTypeGeometry(type) {
	let str = type;
	if (str) {
		if (str.startsWith("area_verde"))
			return "areaverde";
		else if (str.startsWith("area_blu"))
			return "areablu";
		else if (str.startsWith("poligono_area_verde"))
			return "polareaverde";
		else if (str.startsWith("ingressi"))
			return "ingresso";
		else return "";
	} else
		return "";
}


// funzione per la fetch delle geometrie delle aree blu e verdi
async function fetchDataAree(setAreeBlu, setAreeVerdi, comune) {
	try {
		const areeBluResp = await getAllAreeBluFromComune(comune);
		const areeBluJson = await areeBluResp.json();
		const areeVerdiResp = await getAllAreeVerdiFromComune(comune);
		const areeVerdiJson = await areeVerdiResp.json();

		setAreeBlu(areeBluJson || []);
		setAreeVerdi(areeVerdiJson || []);
	} catch (err) {
		console.error("Errore nel fetch dei dati delle aree:", err);
	}
}

// funzione per la fetch del comune di riferimento alla geometria
async function fetchDataComune(setComune, geomWKT) {

	try {
		const comuneResp = await getComuneFromGeometry(geomWKT);
		const comuneJson = await comuneResp.json();

		let comuneValue = "Sconosciuto";
		if (comuneJson.length !== 0) {
			comuneValue = comuneJson[0].comune;
		}

		setComune(comuneValue);
		return comuneValue;

	} catch (err) {
		console.error("Errore nel fetch dei dati del comune:", err);
	}
}

// funzione per la fetch delle geometrie delle aree blu e verdi
async function fetchDataAreaBlu(id, setValueAreaBlu) {
	try {
		const areaBluResp = await getAreaBluFromId(id);
		const areaBluJson = await areaBluResp.json();

		setValueAreaBlu(areaBluJson || null);
	} catch (err) {
		console.error("Errore nel fetch dei dati area blu:", err);
	}
}

// funzione per la fetch delle geometrie delle aree blu e verdi
async function fetchDataAreaVerde(id, setValueAreaVerde) {
	try {
		const areaVerdeResp = await getAreaVerdeFromId(id);
		const areaVerdeJson = await areaVerdeResp.json();

		setValueAreaVerde(areaVerdeJson || null);
	} catch (err) {
		console.error("Errore nel fetch dei dati area verde:", err);
	}
}